# -*- coding: utf-8 -*-
"""
This code generates the figures used in the analysis of the nucleosome model (see Section SI.7).
The model was first introduced in:
Dodd IB, Micheelsen MA, Sneppen K, Thon G. Theoretical analysis of epigenetic cell 
memory by nucleosome modification. Cell. 2007 May 18;129(4):813-22. 
doi: 10.1016/j.cell.2007.02.053. PMID: 17512413.

Created on Mon Apr 22 10:28:59 2024
@author: Andrei Sontag
"""

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import matplotlib
import seaborn as sn

N = 60

stabs = pd.read_csv('stability',sep=',',index_col=0)
curvs = pd.read_csv('curvature',sep=',',index_col=0)

# Figures
# Create layout
layout = [
    ["A", "B"],
    ["C", "D"],
]

matplotlib.rcParams.update({'font.size': 15})

fig, axd = plt.subplot_mosaic(layout, figsize=(10,10))


# Plot the data
# Adjusting the layout for better view
plt.tight_layout(pad=2)
#plt.show()

#cmap = 'viridis'

# 1D arrays
x = np.linspace(0,1,101)
y = np.linspace(0,1,101)

ax = sn.heatmap(stabs,cbar=True,cmap='Blues',ax=axd['A'])
axd['A'].set_title(r'Polarisation')
axd['A'].set_xlabel('p')
axd['A'].set_ylabel(r'$\mathcal{N}$')

ax = sn.heatmap(curvs,cbar=True,cmap='Blues_r',ax=axd['B'])
axd['B'].set_title(r'Curvature')
axd['B'].set_xlabel('p')
axd['B'].set_ylabel(r'$\mathcal{N}$')

file_name = "nuc_%d_%d.txt" % (2,83)
data = np.loadtxt(r'.\nuc\\'+file_name,delimiter=',')

points = [[0, N+1], [N+1, N+1], [N+1, 0]]
triangle = np.array(points+ points[:1])

axd['C'].plot(x*(N+1),(1-x)*(N+1),linestyle='dashed',color='white',linewidth=3)
ax = sn.heatmap((data+np.transpose(data)),cbar=False,ax=axd['C'],cmap='Blues')
axd['C'].invert_yaxis()
#axd['C'].set_title(r'$\mathcal{N} = 2, p = 0.83$')
axd['C'].set_xlabel('Methylated')
axd['C'].set_ylabel('Acetylated')

axd['C'].fill(triangle[:, 0], triangle[:, 1], color='white', alpha=1)

file_name = "nuc_%d_%d.txt" % (20,55)
data = np.loadtxt(r'.\nuc\\'+file_name,delimiter=',')

axd['D'].plot(x*(N+1),(1-x)*(N+1),linestyle='dashed',color='white',linewidth=3)
ax = sn.heatmap((data+np.transpose(data)),cbar=False,ax=axd['D'],cmap='Blues')
axd['D'].invert_yaxis()
#axd['D'].set_title(r'$\mathcal{N} = 20, p = 0.55$')
axd['D'].set_xlabel('Methylated')
axd['D'].set_ylabel('Acetylated')

axd['D'].fill(triangle[:, 0], triangle[:, 1], color='white', alpha=1)

xax = np.linspace(0,N+1,101)

sn.lineplot(x=xax, y=N+1-xax,linestyle='-',color='black',linewidth=2,ax=axd['C'])
sn.lineplot(x=xax, y=len(xax)*[0.1],linestyle='-',color='black',linewidth=2,ax=axd['C'])
axd['C'].plot(len(xax)*[0.1], xax,linestyle='-',color='black',linewidth=2)

sn.lineplot(x=xax, y=N+1-xax,linestyle='-',color='black',linewidth=2,ax=axd['D'])
sn.lineplot(x=xax, y=len(xax)*[0.1],linestyle='-',color='black',linewidth=2,ax=axd['D'])
axd['D'].plot(len(xax)*[0.1], xax,linestyle='-',color='black',linewidth=2)